<?php
/**
 * Hjälpfunktioner för att hantera participants (users och staff)
 */

require_once __DIR__ . '/mapping_helper.php';
require_once __DIR__ . '/dal/repositories/UserRepository.php';
require_once __DIR__ . '/dal/repositories/MappingRepository.php';

/**
 * Bestäm om en participant_id är en user, admin eller extern
 * @param string $participantId
 * @return string|null 'user', 'admin', 'extern', eller null om inte hittad
 */
function getParticipantType($participantId) {
    if (empty($participantId)) {
        return null;
    }
    
    $userRepo = new UserRepository();
    $user = $userRepo->getById($participantId);
    
    if ($user) {
        return $user['user_type'] ?? 'user';
    }
    
    return null;
}

/**
 * Hämta namn för en participant
 * @param string $participantId
 * @return string
 */
function getParticipantName($participantId) {
    if (empty($participantId)) {
        return $participantId;
    }
    
    $userRepo = new UserRepository();
    $user = $userRepo->getById($participantId);
    
    if ($user) {
        // Använd nickname om det finns och inte är tomt, annars full_name
        $nickname = trim($user['nickname'] ?? '');
        if (!empty($nickname)) {
            return $nickname;
        }
        return $user['full_name'] ?? $participantId;
    }
    
    return $participantId;
}

/**
 * Kontrollera om en participant är admin
 * @param string $participantId
 * @return bool
 */
function isParticipantAdmin($participantId) {
    if (empty($participantId)) {
        return false;
    }
    
    $userRepo = new UserRepository();
    $user = $userRepo->getById($participantId);
    
    if ($user) {
        return ($user['user_type'] ?? 'user') === 'admin';
    }
    
    return false;
}

/**
 * Hämta alla admin user_ids
 * @return array Array med admin user_ids
 */
function getAllAdminUserIds() {
    $userRepo = new UserRepository();
    $admins = $userRepo->getByType('admin');
    
    $adminIds = [];
    foreach ($admins as $admin) {
        if (isset($admin['user_id'])) {
            $adminIds[] = $admin['user_id'];
        }
    }
    
    return $adminIds;
}

/**
 * Kontrollera om en participant är aktiv
 * @param string $participantId
 * @return bool
 */
function isParticipantActive($participantId) {
    if (empty($participantId)) {
        return false;
    }
    
    $userRepo = new UserRepository();
    $user = $userRepo->getById($participantId);
    
    if ($user) {
        $active = $user['active'] ?? true;
        return ($active === true || $active === 1 || $active === '1' || $active === 'true');
    }
    
    return false; // Om participant inte hittas, anses de som inaktiva
}

/**
 * Kontrollera om en extern användare kan delta i chatt
 * @param string $participantId
 * @return bool
 */
function canExternalUserChat($participantId) {
    if (empty($participantId)) {
        return false;
    }
    
    $userRepo = new UserRepository();
    $user = $userRepo->getById($participantId);
    
    if ($user) {
        $userType = $user['user_type'] ?? 'user';
        // Endast externa användare behöver denna kontroll
        if ($userType !== 'extern') {
            return true; // Interna användare och admins kan alltid chatta
        }
        // För externa användare, kontrollera can_chat (default true om inte satt)
        $canChat = $user['can_chat'] ?? true;
        return ($canChat === true || $canChat === 1 || $canChat === '1' || $canChat === 'true');
    }
    
    return false;
}

/**
 * Kontrollera om en extern användare kan ladda upp bilder
 * @param string $participantId
 * @return bool
 */
function canExternalUserUpload($participantId) {
    if (empty($participantId)) {
        return false;
    }
    
    $userRepo = new UserRepository();
    $user = $userRepo->getById($participantId);
    
    if ($user) {
        $userType = $user['user_type'] ?? 'user';
        // Endast externa användare behöver denna kontroll
        if ($userType !== 'extern') {
            return true; // Interna användare och admins kan alltid ladda upp
        }
        // För externa användare, kontrollera can_upload (default true om inte satt)
        $canUpload = $user['can_upload'] ?? true;
        return ($canUpload === true || $canUpload === 1 || $canUpload === '1' || $canUpload === 'true');
    }
    
    return false;
}

/**
 * Kontrollera om en användare är i participants-listan för ett projekt
 * @param string $projectId
 * @param string $participantId
 * @param string|null $projectsFile Sökväg till projects.json (optional, används inte längre men behålls för bakåtkompatibilitet)
 * @return bool
 */
function isProjectParticipant($projectId, $participantId, $projectsFile = null) {
    if (empty($projectId) || empty($participantId)) {
        return false;
    }
    
    // Admins är alltid participants
    if (isParticipantAdmin($participantId)) {
        return true;
    }
    
    // Läs från project_mapping
    $participants = readProjectMapping($projectId);
    
    foreach ($participants as $p) {
        $pid = $p['participant_id'] ?? '';
        if ($pid === $participantId) {
            return true;
        }
    }
    
    return false;
}

/**
 * Kontrollera om ett nickname är unikt (inte används av någon annan användare)
 * @param string $nickname Nickname att kontrollera
 * @param string|null $excludeUserId User ID att exkludera från kontrollen (för uppdateringar)
 * @return bool True om nickname är unikt eller tomt, false om det redan används
 */
function isNicknameUnique($nickname, $excludeUserId = null) {
    // Tomt nickname är alltid "unikt" (tillåtet)
    if (empty($nickname) || trim($nickname) === '') {
        return true;
    }
    
    $nickname = trim($nickname);
    $userRepo = new UserRepository();
    $allUsers = $userRepo->getAll();
    
    foreach ($allUsers as $user) {
        $userId = $user['user_id'] ?? '';
        // Hoppa över den användare som exkluderas (vid uppdatering)
        if (!empty($excludeUserId) && $userId === $excludeUserId) {
            continue;
        }
        
        $userNickname = trim($user['nickname'] ?? '');
        // Om nickname matchar (case-insensitive), är det inte unikt
        if (!empty($userNickname) && strcasecmp($userNickname, $nickname) === 0) {
            return false;
        }
    }
    
    return true;
}

/**
 * Kontrollera om en användare är i chatters-listan för ett projekt
 * @param string $projectId
 * @param string $participantId
 * @param string|null $projectsFile Sökväg till projects.json (optional, används inte längre men behålls för bakåtkompatibilitet)
 * @return bool
 */
function isProjectChatter($projectId, $participantId, $projectsFile = null) {
    if (empty($projectId) || empty($participantId)) {
        return false;
    }
    
    // Admins är alltid chatters
    if (isParticipantAdmin($participantId)) {
        return true;
    }
    
    // Läs från chat_mapping
    $chatters = readChatMapping($projectId);
    
    foreach ($chatters as $c) {
        $cid = $c['participant_id'] ?? '';
        if ($cid === $participantId) {
            return true;
        }
    }
    
    return false;
}

/**
 * Kontrollera om en användare kan chatta i ett projekt
 * Kombinerar can_chat (master switch) och chatters-listan
 * @param string $projectId
 * @param string $participantId
 * @param string|null $projectsFile Sökväg till projects.json (optional, används inte längre men behålls för bakåtkompatibilitet)
 * @return bool
 */
function canUserChat($projectId, $participantId, $projectsFile = null) {
    if (empty($projectId) || empty($participantId)) {
        return false;
    }
    
    // Admins kan alltid chatta
    if (isParticipantAdmin($participantId)) {
        return true;
    }
    
    $userType = getParticipantType($participantId);
    
    // För interna användare (user/admin), kan de alltid chatta
    if ($userType !== 'extern') {
        return true;
    }
    
    // För externa användare: kontrollera can_chat (master switch)
    if (!canExternalUserChat($participantId)) {
        return false; // Om can_chat är false, kan de inte chatta oavsett chatters-lista
    }
    
    // Om can_chat är true, kontrollera chatters-listan
    return isProjectChatter($projectId, $participantId, $projectsFile);
}

/**
 * Kontrollera om en användare kan ladda upp till ett projekt
 * Kombinerar can_upload (master switch) och participants-listan
 * @param string $projectId
 * @param string $participantId
 * @param string|null $projectsFile Sökväg till projects.json (optional, används inte längre men behålls för bakåtkompatibilitet)
 * @return bool
 */
function canUserUpload($projectId, $participantId, $projectsFile = null) {
    if (empty($projectId) || empty($participantId)) {
        return false;
    }
    
    // Admins kan alltid ladda upp
    if (isParticipantAdmin($participantId)) {
        return true;
    }
    
    $userType = getParticipantType($participantId);
    
    // För interna användare (user/admin), kan de alltid ladda upp
    if ($userType !== 'extern') {
        return true;
    }
    
    // För externa användare: kontrollera can_upload (master switch)
    if (!canExternalUserUpload($participantId)) {
        return false; // Om can_upload är false, kan de inte ladda upp oavsett participants-lista
    }
    
    // Om can_upload är true, kontrollera participants-listan
    return isProjectParticipant($projectId, $participantId, $projectsFile);
}

