<?php
/**
 * UUID generation and management functions
 */

/**
 * Generate a UUID v4 (random)
 */
function generateUUID() {
    // Use PHP's built-in function if available (PHP 7.0+)
    if (function_exists('random_bytes')) {
        $data = random_bytes(16);
    } else {
        // Fallback for older PHP versions
        $data = openssl_random_pseudo_bytes(16);
    }
    
    // Set version (4) and variant bits
    $data[6] = chr(ord($data[6]) & 0x0f | 0x40); // Version 4
    $data[8] = chr(ord($data[8]) & 0x3f | 0x80); // Variant bits
    
    // Format as UUID string
    return vsprintf('%s%s-%s-%s-%s-%s%s%s', str_split(bin2hex($data), 4));
}

/**
 * Validate UUID format
 */
function isValidUUID($uuid) {
    return preg_match('/^[0-9a-f]{8}-[0-9a-f]{4}-4[0-9a-f]{3}-[89ab][0-9a-f]{3}-[0-9a-f]{12}$/i', $uuid);
}

/**
 * Generate a short UUID for display purposes (first 8 characters)
 */
function generateShortUUID($uuid = null) {
    if ($uuid === null) {
        $uuid = generateUUID();
    }
    return substr(str_replace('-', '', $uuid), 0, 8);
}

/**
 * Create a user-friendly link ID (shorter than full UUID)
 */
function generateUserLinkId($uuid = null) {
    return generateShortUUID($uuid);
}
?>
