<?php
/**
 * Centraliserad logger för server-side loggning
 * 
 * Alla loggar skrivs till /logs mappen med separata filer per domän
 * 
 * Användning:
 *   require_once __DIR__ . '/logger.php';
 *   logMessage('config', 'Meddelande', ['context' => 'data']);
 *   logMessage('api', 'API-anrop', ['endpoint' => '/api/chat/messages.php']);
 * 
 * @param string $domain Användningsdomän (t.ex. 'config', 'api', 'security', 'external_staff')
 * @param string $message Loggmeddelande
 * @param array $context Ytterligare kontextdata (valfritt)
 * @param string $level Loggnivå ('info', 'warning', 'error', 'debug') - standard är 'info'
 */
function logMessage($domain, $message, $context = [], $level = 'info') {
    // Validera level
    $validLevels = ['info', 'warning', 'error', 'debug'];
    if (!in_array(strtolower($level), $validLevels)) {
        $level = 'info';
    }
    
    // Säkerställ att domain är ett giltigt filnamn
    $domain = preg_replace('/[^a-zA-Z0-9_-]/', '_', $domain);
    if (empty($domain)) {
        $domain = 'general';
    }
    
    // Bestäm loggfilens sökväg
    $logDir = __DIR__ . '/logs';
    $logFile = $logDir . '/' . $domain . '.log';
    
    // Skapa logs-mappen om den inte finns
    if (!is_dir($logDir)) {
        @mkdir($logDir, 0755, true);
    }
    
    // Bygg loggpost
    $timestamp = date('Y-m-d H:i:s');
    $ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
    
    $logEntry = [
        'timestamp' => $timestamp,
        'level' => strtolower($level),
        'ip' => $ip,
        'message' => $message
    ];
    
    // Lägg till kontext om det finns
    if (!empty($context)) {
        $logEntry['context'] = $context;
    }
    
    // Lägg till request-information om det finns
    if (isset($_SERVER['REQUEST_URI'])) {
        $logEntry['request_uri'] = $_SERVER['REQUEST_URI'];
    }
    if (isset($_SERVER['REQUEST_METHOD'])) {
        $logEntry['request_method'] = $_SERVER['REQUEST_METHOD'];
    }
    
    // Skriv till loggfil
    $logLine = json_encode($logEntry, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES) . "\n";
    @file_put_contents($logFile, $logLine, FILE_APPEND | LOCK_EX);
}

/**
 * Hjälpfunktioner för olika loggnivåer
 */
function logInfo($domain, $message, $context = []) {
    logMessage($domain, $message, $context, 'info');
}

function logWarning($domain, $message, $context = []) {
    logMessage($domain, $message, $context, 'warning');
}

function logError($domain, $message, $context = []) {
    logMessage($domain, $message, $context, 'error');
}

function logDebug($domain, $message, $context = []) {
    logMessage($domain, $message, $context, 'debug');
}
?>

