<?php
/**
 * rights.php - Sätter rätt rättigheter för webbservern
 * Körs en gång för att fixa rättigheter på Synology NAS
 */

echo "<h1>Inställning av rättigheter för ByggArkiv</h1>\n";

// Kontrollera om vi kör som root (sudo)
if (posix_getuid() !== 0) {
    echo "<p style='color: red;'>VARNING: Denna fil måste köras som root (sudo) för att kunna ändra rättigheter.</p>\n";
    echo "<p>Kör: sudo php rights.php</p>\n";
    exit(1);
}

echo "<p>Körs som root - fortsätter med att sätta rättigheter...</p>\n";

// Lista över filer och mappar som behöver rättigheter
$filesToFix = [
    'data/',
    'uploads/',
    'config.php',
    'upload.php',
    'check_user_status.php'
];

$successCount = 0;
$errorCount = 0;

foreach ($filesToFix as $path) {
    echo "<h3>Hanterar: $path</h3>\n";
    
    if (!file_exists($path)) {
        echo "<p style='color: orange;'>Filen/mappen finns inte: $path</p>\n";
        continue;
    }
    
    try {
        // Sätt ägare till webadmin
        if (chown($path, 'webadmin')) {
            echo "<p>✓ Ägare satt till webadmin</p>\n";
        } else {
            echo "<p style='color: orange;'>⚠ Kunde inte sätta ägare till webadmin</p>\n";
        }
        
        // Sätt grupp till http
        if (chgrp($path, 'http')) {
            echo "<p>✓ Grupp satt till http</p>\n";
        } else {
            echo "<p style='color: orange;'>⚠ Kunde inte sätta grupp till http</p>\n";
        }
        
        // Sätt rättigheter
        if (is_dir($path)) {
            // Mappar: 775 (ägare och grupp kan läsa/skriva/köra, andra kan läsa/köra)
            if (chmod($path, 0775)) {
                echo "<p>✓ Rättigheter satta till 775 (mapp)</p>\n";
            } else {
                echo "<p style='color: red;'>✗ Kunde inte sätta rättigheter till 775</p>\n";
                $errorCount++;
                continue;
            }
        } else {
            // Filer: 664 (ägare och grupp kan läsa/skriva, andra kan läsa)
            if (chmod($path, 0664)) {
                echo "<p>✓ Rättigheter satta till 664 (fil)</p>\n";
            } else {
                echo "<p style='color: red;'>✗ Kunde inte sätta rättigheter till 664</p>\n";
                $errorCount++;
                continue;
            }
        }
        
        // Kontrollera resultatet
        $perms = fileperms($path);
        $owner = posix_getpwuid(fileowner($path))['name'] ?? 'unknown';
        $group = posix_getgrgid(filegroup($path))['name'] ?? 'unknown';
        
        echo "<p>Resultat: ägare=$owner, grupp=$group, rättigheter=" . substr(sprintf('%o', $perms), -3) . "</p>\n";
        
        $successCount++;
        
    } catch (Exception $e) {
        echo "<p style='color: red;'>✗ Fel vid hantering av $path: " . $e->getMessage() . "</p>\n";
        $errorCount++;
    }
}

// Hantera uploads-mappen rekursivt
echo "<h3>Hanterar uploads-mappen rekursivt</h3>\n";
if (is_dir('uploads/')) {
    $iterator = new RecursiveIteratorIterator(
        new RecursiveDirectoryIterator('uploads/', RecursiveDirectoryIterator::SKIP_DOTS),
        RecursiveIteratorIterator::SELF_FIRST
    );
    
    foreach ($iterator as $file) {
        $filePath = $file->getPathname();
        try {
            chown($filePath, 'webadmin');
            chgrp($filePath, 'http');
            
            if ($file->isDir()) {
                chmod($filePath, 0775);
            } else {
                chmod($filePath, 0664);
            }
            
            echo "<p>✓ $filePath</p>\n";
            $successCount++;
        } catch (Exception $e) {
            echo "<p style='color: red;'>✗ $filePath: " . $e->getMessage() . "</p>\n";
            $errorCount++;
        }
    }
}

// Sammanfattning
echo "<h2>Sammanfattning</h2>\n";
echo "<p>Filer/mappar hanterade: " . count($filesToFix) . "</p>\n";
echo "<p style='color: green;'>✓ Framgångsrika: $successCount</p>\n";
if ($errorCount > 0) {
    echo "<p style='color: red;'>✗ Fel: $errorCount</p>\n";
}

echo "<h3>Nästa steg</h3>\n";
echo "<p>1. Testa att ladda upp en bild med koden '1234'</p>\n";
echo "<p>2. Om det fortfarande inte fungerar, kontrollera att webbservern körs under användaren 'http'</p>\n";
echo "<p>3. Kontrollera att 'webadmin'-användaren finns på systemet</p>\n";

// Visa aktuella rättigheter för viktiga filer
echo "<h3>Aktuella rättigheter</h3>\n";
foreach ($filesToFix as $path) {
    if (file_exists($path)) {
        $perms = fileperms($path);
        $owner = posix_getpwuid(fileowner($path))['name'] ?? 'unknown';
        $group = posix_getgrgid(filegroup($path))['name'] ?? 'unknown';
        $permStr = substr(sprintf('%o', $perms), -3);
        
        echo "<p><strong>$path</strong>: ägare=$owner, grupp=$group, rättigheter=$permStr</p>\n";
    }
}
?>
