<?php
/**
 * Säkerhetsfunktioner för ByggArkiv (utan admin-inloggning)
 */

// Input sanitering
function sanitizeInput($input, $type = 'string') {
    if (is_null($input)) return null;
    
    switch ($type) {
        case 'string':
            return htmlspecialchars(trim($input), ENT_QUOTES, 'UTF-8');
        case 'filename':
            // Tillåt endast alfanumeriska tecken, bindestreck, understreck och punkter
            return preg_replace('/[^a-zA-Z0-9._-]/', '', $input);
        case 'project_id':
            // Tillåt endast alfanumeriska tecken och bindestreck
            return preg_replace('/[^a-zA-Z0-9-]/', '', $input);
        case 'user_id':
            // Tillåt endast alfanumeriska tecken, bindestreck och understreck
            return preg_replace('/[^a-zA-Z0-9_-]/', '', $input);
        case 'int':
            return filter_var($input, FILTER_VALIDATE_INT);
        default:
            return htmlspecialchars(trim($input), ENT_QUOTES, 'UTF-8');
    }
}

// Rate limiting för uploads
function checkUploadRateLimit($identifier, $maxAttempts = 25, $timeWindow = 300) {
    $key = 'upload_rate_limit_' . md5($identifier);
    
    // Läs befintlig data från fil
    $rateLimitFile = __DIR__ . '/../data/upload_limits.json';
    $rateLimits = [];
    if (file_exists($rateLimitFile)) {
        $rateLimits = json_decode(file_get_contents($rateLimitFile), true) ?: [];
    }
    
    // Skapa data-mappen om den inte finns
    $dataDir = dirname($rateLimitFile);
    if (!is_dir($dataDir)) {
        mkdir($dataDir, 0755, true);
    }
    
    if (!isset($rateLimits[$key])) {
        $rateLimits[$key] = ['count' => 0, 'first_attempt' => time(), 'user_id' => $identifier];
    }
    
    $data = $rateLimits[$key];
    
    // Reset om tidsfönstret har passerat
    if (time() - $data['first_attempt'] > $timeWindow) {
        $rateLimits[$key] = ['count' => 1, 'first_attempt' => time(), 'user_id' => $identifier];
    } else {
        // Kontrollera om gränsen överskrids
        if ($data['count'] >= $maxAttempts) {
            // Logga rate limit-träff
            logSecurityEvent('rate_limit_exceeded', [
                'identifier' => $identifier,
                'count' => $data['count'],
                'max_attempts' => $maxAttempts,
                'time_window' => $timeWindow
            ], 'low', $identifier);
            return false;
        }
        
        $rateLimits[$key]['count']++;
    }
    
    // Kontrollera om gränsen överskrids efter uppdatering
    if ($rateLimits[$key]['count'] > $maxAttempts) {
        // Logga rate limit-träff
        logSecurityEvent('rate_limit_exceeded', [
            'identifier' => $identifier,
            'count' => $rateLimits[$key]['count'],
            'max_attempts' => $maxAttempts,
            'time_window' => $timeWindow
        ], 'low', $identifier);
        return false;
    }
    
    // Spara till fil
    file_put_contents($rateLimitFile, json_encode($rateLimits, JSON_PRETTY_PRINT));
    
    return true;
}

// Säker filuppladdning för bilder
function validateUploadedImage($file) {
    $errors = [];
    
    // Kontrollera upload error
    if ($file['error'] !== UPLOAD_ERR_OK) {
        $errors[] = 'Upload error: ' . $file['error'];
        return $errors;
    }
    
    // Kontrollera filstorlek
    if ($file['size'] > MAX_FILE_BYTES) {
        $fileSizeMB = round($file['size'] / (1024 * 1024), 1);
        $maxSizeMB = MAX_FILE_BYTES / (1024 * 1024);
        $errors[] = "Bilden är för stor: {$fileSizeMB} MB (max {$maxSizeMB} MB). Kontakta administratören för att ladda upp större bilder.";
    }
    
    // Kontrollera filändelse först (för att ge bättre felmeddelanden)
    $extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
    global $ALLOWED_IMAGE_MIME;
    $allowedExtensions = array_values($ALLOWED_IMAGE_MIME);
    // Lägg till jpeg som alias för jpg
    if ($extension === 'jpeg') {
        $extension = 'jpg';
    }
    
    // Ge specifikt felmeddelande för SVG
    if ($extension === 'svg') {
        $errors[] = 'SVG-filer stöds inte. Använd JPG, PNG eller WebP istället.';
        return $errors; // Returnera tidigt för SVG
    }
    
    if (!in_array($extension, $allowedExtensions)) {
        $errors[] = 'Otillåten filändelse: ' . $extension . '. Tillåtna format: ' . implode(', ', $allowedExtensions);
    }
    
    // Kontrollera MIME-typ för bilder
    $finfo = new finfo(FILEINFO_MIME_TYPE);
    $mime = $finfo->file($file['tmp_name']);
    
    if (!isset($ALLOWED_IMAGE_MIME[$mime])) {
        $errors[] = 'Otillåten bildtyp: ' . $mime;
    }
    
    // Kontrollera att filen verkligen är en bild (fungerar inte för SVG)
    $imageInfo = @getimagesize($file['tmp_name']);
    if ($imageInfo === false) {
        $errors[] = 'Filen är inte en giltig bild';
    }
    
    return $errors;
}

// Säker filuppladdning för dokument
function validateUploadedDocument($file) {
    $errors = [];
    
    // Kontrollera upload error
    if ($file['error'] !== UPLOAD_ERR_OK) {
        $errors[] = 'Upload error: ' . $file['error'];
        return $errors;
    }
    
    // Kontrollera filstorlek
    if ($file['size'] > MAX_FILE_BYTES) {
        $fileSizeMB = round($file['size'] / (1024 * 1024), 1);
        $maxSizeMB = MAX_FILE_BYTES / (1024 * 1024);
        $errors[] = "Dokumentet är för stort: {$fileSizeMB} MB (max {$maxSizeMB} MB). Kontakta administratören för att ladda upp större dokument.";
    }
    
    // Kontrollera MIME-typ för dokument
    $finfo = new finfo(FILEINFO_MIME_TYPE);
    $mime = $finfo->file($file['tmp_name']);
    global $ALLOWED_DOCUMENT_MIME;
    
    if (!isset($ALLOWED_DOCUMENT_MIME[$mime])) {
        $errors[] = 'Otillåten dokumenttyp: ' . $mime;
    }
    
    // Kontrollera filändelse
    $extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
    $allowedExtensions = array_values($ALLOWED_DOCUMENT_MIME);
    if (!in_array($extension, $allowedExtensions)) {
        $errors[] = 'Otillåten filändelse: ' . $extension;
    }
    
    return $errors;
}

// Säker filuppladdning för videor
function validateUploadedVideo($file) {
    $errors = [];
    
    // Kontrollera upload error
    if ($file['error'] !== UPLOAD_ERR_OK) {
        $errors[] = 'Upload error: ' . $file['error'];
        return $errors;
    }
    
    // Kontrollera filstorlek (använd större gräns för videor)
    if ($file['size'] > MAX_VIDEO_BYTES) {
        $fileSizeMB = round($file['size'] / (1024 * 1024), 1);
        $maxSizeMB = MAX_VIDEO_BYTES / (1024 * 1024);
        $errors[] = "Videofilen är för stor: {$fileSizeMB} MB (max {$maxSizeMB} MB). Kontakta administratören för att ladda upp större videor.";
    }
    
    // Kontrollera MIME-typ för videor
    $finfo = new finfo(FILEINFO_MIME_TYPE);
    $mime = $finfo->file($file['tmp_name']);
    global $ALLOWED_VIDEO_MIME;
    
    if (!isset($ALLOWED_VIDEO_MIME[$mime])) {
        $errors[] = 'Otillåten videotyp: ' . $mime;
    }
    
    // Kontrollera filändelse
    $extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
    $allowedExtensions = array_values($ALLOWED_VIDEO_MIME);
    if (!in_array($extension, $allowedExtensions)) {
        $errors[] = 'Otillåten filändelse: ' . $extension;
    }
    
    return $errors;
}

// Universal filvalidering som hanterar bilder, dokument och videor
function validateUploadedFile($file) {
    $errors = [];
    
    // Kontrollera upload error
    if ($file['error'] !== UPLOAD_ERR_OK) {
        $errors[] = 'Upload error: ' . $file['error'];
        return $errors;
    }
    
    // Bestäm filtyp baserat på MIME-typ
    $finfo = new finfo(FILEINFO_MIME_TYPE);
    $mime = $finfo->file($file['tmp_name']);
    
    global $ALLOWED_IMAGE_MIME, $ALLOWED_DOCUMENT_MIME, $ALLOWED_VIDEO_MIME;
    
    if (isset($ALLOWED_IMAGE_MIME[$mime])) {
        // Det är en bild - använd bildvalidering
        return validateUploadedImage($file);
    } elseif (isset($ALLOWED_DOCUMENT_MIME[$mime])) {
        // Det är ett dokument - använd dokumentvalidering
        return validateUploadedDocument($file);
    } elseif (isset($ALLOWED_VIDEO_MIME[$mime])) {
        // Det är en video - använd videovalidering
        return validateUploadedVideo($file);
    } else {
        // Okänd filtyp
        $errors[] = 'Otillåten filtyp: ' . $mime;
        return $errors;
    }
}

// Säker filnamn generering
function generateSecureFilename($originalName, $projectId) {
    $extension = strtolower(pathinfo($originalName, PATHINFO_EXTENSION));
    $timestamp = date('Ymd_His');
    $random = bin2hex(random_bytes(4));
    
    return $timestamp . '_' . $random . '.' . $extension;
}

// Säkerhetsheaders
function setSecurityHeaders() {
    // Prevent XSS
    header('X-Content-Type-Options: nosniff');
    header('X-Frame-Options: DENY');
    header('X-XSS-Protection: 1; mode=block');
    
    // Content Security Policy
    header("Content-Security-Policy: default-src 'self'; script-src 'self' 'unsafe-inline'; style-src 'self' 'unsafe-inline'; img-src 'self' data:;");
    
    // Referrer Policy
    header('Referrer-Policy: strict-origin-when-cross-origin');
    
    // Remove server information
    header_remove('X-Powered-By');
    header_remove('Server');
}

// Loggning av säkerhetshändelser
// Severity-nivåer: 'low', 'medium', 'high', 'critical'
function logSecurityEvent($event, $details = [], $severity = 'medium', $userId = null, $projectId = null, $action = null) {
    $logFile = __DIR__ . '/../logs/security.log';
    $timestamp = date('Y-m-d H:i:s');
    $ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
    $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? 'unknown';
    
    // Validera severity
    $validSeverities = ['low', 'medium', 'high', 'critical'];
    if (!in_array($severity, $validSeverities)) {
        $severity = 'medium';
    }
    
    $logEntry = [
        'timestamp' => $timestamp,
        'event_type' => $event,
        'severity' => $severity,
        'ip' => $ip,
        'user_agent' => $userAgent,
        'details' => $details
    ];
    
    // Lägg till kontext om det finns
    if ($userId !== null) {
        $logEntry['user_id'] = $userId;
    }
    if ($projectId !== null) {
        $logEntry['project_id'] = $projectId;
    }
    if ($action !== null) {
        $logEntry['action'] = $action;
    }
    
    // Skapa logs-mappen om den inte finns
    $logDir = dirname($logFile);
    if (!is_dir($logDir)) {
        mkdir($logDir, 0755, true);
    }
    
    file_put_contents($logFile, json_encode($logEntry, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES) . "\n", FILE_APPEND | LOCK_EX);
}

// Generell loggfunktion för att logga till logs-mappen
function writeLog($logName, $message, $context = []) {
    $logFile = __DIR__ . '/../logs/' . $logName . '.log';
    $logDir = dirname($logFile);
    
    // Skapa logs-mappen om den inte finns
    if (!is_dir($logDir)) {
        @mkdir($logDir, 0755, true);
    }
    
    $timestamp = date('Y-m-d H:i:s');
    $ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
    
    $logEntry = [
        'timestamp' => $timestamp,
        'ip' => $ip,
        'message' => $message
    ];
    
    if (!empty($context)) {
        $logEntry['context'] = $context;
    }
    
    $logLine = json_encode($logEntry, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES) . "\n";
    @file_put_contents($logFile, $logLine, FILE_APPEND | LOCK_EX);
}
?>
