<?php
// Starta output buffering tidigt för att förhindra att varningar skrivs ut
ob_start();

// Förhindra att PHP-varningar skrivs ut (de loggas fortfarande)
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/lib/data_functions.php';
require_once __DIR__ . '/lib/uuid_functions.php';
require_once __DIR__ . '/security.php';
require_once __DIR__ . '/lib/participant_helper.php';

// Säkerhetsheaders
setSecurityHeaders();

// Rensa output buffer (tar bort eventuella varningar)
if (ob_get_level() > 0) {
    ob_clean();
}

// Svara alltid JSON
header('Content-Type: application/json; charset=utf-8');

// Kontrollera POST-storlek tidigt för användarvänliga felmeddelanden
$maxPostSize = 32 * 1024 * 1024; // 32MB server limit
$contentLength = $_SERVER['CONTENT_LENGTH'] ?? 0;

if ($contentLength > $maxPostSize) {
    $fileSizeMB = round($contentLength / (1024 * 1024), 1);
    $maxSizeMB = $maxPostSize / (1024 * 1024);
    http_response_code(413);
    echo json_encode([
        'ok' => false, 
        'error' => "Filen är för stor: {$fileSizeMB} MB (max {$maxSizeMB} MB). Kontakta administratören för att ladda upp större filer.",
        'file_size_mb' => $fileSizeMB,
        'max_size_mb' => $maxSizeMB
    ]);
    exit;
}

// Rate limiting för uploads - per användare
$userId = sanitizeInput($_POST['user_id'] ?? $_SESSION['user_id'] ?? '', 'user_id');
if (!empty($userId) && !checkUploadRateLimit($userId . '_upload', 25, 300)) { // Max 25 uploads per användare per 5 minuter
  
    http_response_code(429);
    echo json_encode(['ok' => false, 'error' => 'För många uppladdningar. Försök igen senare.']);
    exit;
}

// Funktion för att kontrollera om användare är aktiv
function isUserActive($userId) {
  if (empty($userId)) {
    return false;
  }
  
  $userIdsFile = __DIR__ . '/data/userids.json';
  if (!file_exists($userIdsFile)) {
    return false;
  }
  
  $content = file_get_contents($userIdsFile);
  if ($content === false) {
    return false;
  }
  
  $userIds = json_decode($content, true);
  if (!is_array($userIds)) {
    return false;
  }
  
  foreach ($userIds as $user) {
    if (($user['user_id'] ?? '') === $userId) {
      // För externa användare, kontrollera 'active' fältet
      $userType = $user['user_type'] ?? 'user';
      if ($userType === 'extern') {
        $active = $user['active'] ?? true;
        return ($active === true || $active === 1 || $active === '1' || $active === 'true');
      }
      // För interna användare, kontrollera 'inactive' fältet
      return !isset($user['inactive']) || !$user['inactive'];
    }
  }
  
  return false;
}

// PIN-kod för användaruppladdningar är nu inaktiverad
// Admin-lösenord hanteras via admin_users.json

// Validera fält med sanitering
$project_id = sanitizeInput($_POST['project_id'] ?? '', 'project_id');
$note      = sanitizeInput($_POST['note'] ?? '', 'string');
// $userId redan definierad ovan för rate limiting

// Check if userId is actually a short ID and resolve it (for backward compatibility)
if (!empty($userId) && !isValidUUID($userId)) {
    // If it's not a UUID, try to resolve it as a short ID (for old users)
    $resolvedUserId = resolveLinkToUserId($userId);
    if ($resolvedUserId !== null) {
        $userId = $resolvedUserId;
    }
}

if ($project_id === '') {
  http_response_code(400);
  echo json_encode(['ok' => false, 'error' => 'project_id saknas.', 'debug' => 'Empty project_id']);
  exit;
}

// Kontrollera om användare är aktiv (om user_id finns)
if (!empty($userId) && !isUserActive($userId)) {
  http_response_code(403);
  echo json_encode(['ok' => false, 'error' => 'Användaren är inaktiv och kan inte ladda upp bilder.']);
  exit;
}

// Kontrollera uppladdningstillgång (använder canUserUpload som kontrollerar can_upload OCH participants)
if (!empty($userId)) {
  $projectsFile = __DIR__ . '/data/projects.json';
  if (!canUserUpload($project_id, $userId, $projectsFile)) {
    http_response_code(403);
    echo json_encode(['ok' => false, 'error' => 'Du har inte behörighet att ladda upp bilder till detta projekt.']);
    exit;
  }
}

if (!isset($_FILES['photo']) || $_FILES['photo']['error'] !== UPLOAD_ERR_OK) {
  http_response_code(400);
  $uploadError = isset($_FILES['photo']) ? $_FILES['photo']['error'] : 'No file uploaded';
  echo json_encode(['ok' => false, 'error' => 'Ingen bild mottagen eller upload-fel.', 'debug' => 'File upload error: ' . $uploadError]);
  exit;
}

$file = $_FILES['photo'];

// Säker filvalidering
$validationErrors = validateUploadedFile($file);
if (!empty($validationErrors)) {
    http_response_code(400);
    // Bestäm severity baserat på typ av fel
    $severity = 'medium';
    foreach ($validationErrors as $error) {
        if (stripos($error, 'otillåten') !== false || stripos($error, 'invalid') !== false) {
            $severity = 'high';
            break;
        }
    }
    logSecurityEvent('invalid_upload_attempt', [
        'errors' => $validationErrors,
        'filename' => $file['name'],
        'size' => $file['size']
    ], $severity, $userId, $project_id, 'upload_file');
    echo json_encode(['ok' => false, 'error' => 'Ogiltig fil: ' . implode(', ', $validationErrors)]);
    exit;
}

// Bestäm filändelse säkert
$finfo = new finfo(FILEINFO_MIME_TYPE);
$mime = $finfo->file($file['tmp_name']);
global $ALLOWED_MIME;

$ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
if (isset($ALLOWED_MIME[$mime])) {
    $detectedExt = $ALLOWED_MIME[$mime];
} else {
    // Fallback för iOS HEIC-filer
    $detectedExt = $ext;
}

// Säkerställ projektmapp
$projectDir = __DIR__ . '/' . rtrim(UPLOAD_ROOT, '/\\') . DIRECTORY_SEPARATOR . basename($project_id);
$projectCreated = false;

if (!is_dir($projectDir)) {
  if (!mkdir($projectDir, 0775, true)) {
    http_response_code(500);
    echo json_encode(['ok' => false, 'error' => 'Kan inte skapa projektmapp.']);
    exit;
  }
  $projectCreated = true;
}

// Om projektmappen skapades, lägg till projektet i projects.json
if ($projectCreated) {
  $projectsFile = __DIR__ . '/data/projects.json';
  $projects = [];
  
  if (file_exists($projectsFile)) {
    $projects = json_decode(file_get_contents($projectsFile), true) ?: [];
  }
  
  // Kontrollera om projektet redan finns
  $projectExists = false;
  foreach ($projects as $project) {
    if ($project['id'] === $project_id) {
      $projectExists = true;
      break;
    }
  }
  
  // Lägg till nytt projekt om det inte finns
  if (!$projectExists) {
    $newProject = [
      'id' => $project_id,
      'name' => 'Projekt ' . $project_id,
      'status' => 'active',
      'created_at' => date('Y-m-d H:i:s')
    ];
    
    $projects[] = $newProject;
    
    if (file_put_contents($projectsFile, json_encode($projects, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE)) === false) {
      // Varning om att projects.json inte kunde uppdateras
      error_log("Kunde inte uppdatera projects.json för nytt projekt: $project_id");
    }
  }
}

// Skapa säkert filnamn
$base = generateSecureFilename($file['name'], $project_id);
$base = pathinfo($base, PATHINFO_FILENAME); // Ta bort extension för att lägga till rätt en

$photoPath = $projectDir . DIRECTORY_SEPARATOR . $base . '.' . $detectedExt;
$metaPath  = $projectDir . DIRECTORY_SEPARATOR . $base . '.json';

// Flytta uppfilad fil
if (!move_uploaded_file($file['tmp_name'], $photoPath)) {
  http_response_code(500);
  echo json_encode(['ok' => false, 'error' => 'Kunde inte spara bildfil.', 'debug' => 'Failed to move file from ' . $file['tmp_name'] . ' to ' . $photoPath]);
  exit;
}

// Skapa tumnagel (thumbnail) för snabbare listvisning
// Sparas som JPG: <basename>-thumb.jpg
$thumbPath = $projectDir . DIRECTORY_SEPARATOR . $base . '-thumb.jpg';
$thumbCreated = false;
$thumbError = null;

// Kontrollera om det är en video (hoppa över thumbnail för videor)
$isVideo = false;
global $ALLOWED_VIDEO_MIME;
if (isset($ALLOWED_VIDEO_MIME[$mime])) {
    $isVideo = true;
}

/**
 * Skapar en tumnagel som JPG med maxbredd 320px
 * Använder GD (JPEG/PNG/WebP). HEIC/HEIF hoppas över om inte stöds.
 */
function createThumbnailImage($sourcePath, $destPath, $maxWidth = 320) {
  // GD fallback
  $info = @getimagesize($sourcePath);
  if ($info === false) {
    return false;
  }
  $mime = $info['mime'] ?? '';
  switch ($mime) {
    case 'image/jpeg':
    case 'image/jpg':
    case 'image/pjpeg':
      $src = @imagecreatefromjpeg($sourcePath);
      break;
    case 'image/png':
      $src = @imagecreatefrompng($sourcePath);
      break;
    case 'image/webp':
      if (function_exists('imagecreatefromwebp')) {
        $src = @imagecreatefromwebp($sourcePath);
      } else {
        $src = false;
      }
      break;
    default:
      // GD stödjer ej HEIC/HEIF – hoppa över
      $src = false;
  }
  if (!$src) {
    return false;
  }
  $width = imagesx($src);
  $height = imagesy($src);
  if ($width <= 0 || $height <= 0) {
    imagedestroy($src);
    return false;
  }
  $targetWidth = $width > $maxWidth ? $maxWidth : $width;
  $targetHeight = (int)round(($targetWidth / $width) * $height);
  $dst = imagecreatetruecolor($targetWidth, $targetHeight);
  // För PNG med alfa – fyll vit bakgrund
  $white = imagecolorallocate($dst, 255, 255, 255);
  imagefill($dst, 0, 0, $white);
  imagecopyresampled($dst, $src, 0, 0, 0, 0, $targetWidth, $targetHeight, $width, $height);
  $ok = imagejpeg($dst, $destPath, 75);
  imagedestroy($dst);
  imagedestroy($src);
  return $ok;
}

// Skapa thumbnail endast för bilder, inte för videor
if (!$isVideo) {
  try {
    $thumbCreated = createThumbnailImage($photoPath, $thumbPath, 320);
    if (!$thumbCreated) {
      // Diagnostik: försök avgöra varför
      if (!function_exists('gd_info')) {
        $thumbError = 'GD is not enabled in PHP profile';
      } else {
        // Hämta MIME från tidigare beräkning
        if (in_array($mime, ['image/heic', 'image/heif'], true)) {
          $thumbError = 'HEIC/HEIF not supported by GD on this server';
        } else {
          $thumbError = 'Unsupported format for GD or missing JPEG/PNG/WebP support';
        }
      }
    }
  } catch (Throwable $e) {
    $thumbCreated = false;
    $thumbError = 'Thumbnail generation exception: ' . $e->getMessage();
  }
} else {
  // För videor, hoppa över thumbnail-generering
  $thumbError = 'Thumbnail not generated for video files';
}

// Spara metadata (projekt, anteckning, originalnamn, tid, user_id)
$meta = [
  'project_id'   => $project_id,
  'note'        => $note,
  'user_id'     => $userId, // Lägg till user_id i metadata
  'original'    => $file['name'],
  'mime'        => $mime,
  'bytes'       => $file['size'],
  'stored_file' => basename($photoPath),
  'stored_at'   => date('c'),
  'upload_id'   => time(),
  'file_type'   => $isVideo ? 'video' : 'image', // Lägg till filtyp i metadata
];

if ($thumbCreated && file_exists($thumbPath)) {
  $meta['thumb_file'] = basename($thumbPath);
}

if (file_put_contents($metaPath, json_encode($meta, JSON_PRETTY_PRINT|JSON_UNESCAPED_UNICODE)) === false) {
  // Om metadata inte går att skriva: fortfarande ok att ha sparat filen.
  // Men meddela varning.
  $fileTypeText = $isVideo ? 'Video' : 'Bild';
  echo json_encode(['ok' => true, 'warning' => $fileTypeText . ' sparad men metadata kunde inte sparas.', 'thumb' => $meta['thumb_file'] ?? null, 'thumb_error' => $thumbError]);
  exit;
}

// Tilldela bildnummer
$imageNumber = assignImageNumber($photoPath, $metaPath);
if ($imageNumber !== false) {
  $meta['image_number'] = $imageNumber;
  // Uppdatera metadata med bildnummer
  file_put_contents($metaPath, json_encode($meta, JSON_PRETTY_PRINT|JSON_UNESCAPED_UNICODE));
}

echo json_encode(['ok' => true, 'file' => basename($photoPath), 'upload_id' => time(), 'thumb' => $meta['thumb_file'] ?? null, 'thumb_error' => $thumbError, 'image_number' => $imageNumber]);