<?php
/**
 * Serve files from uploads directory with authentication
 * Serverar filer från uploads/-katalogen med autentisering
 */

// Start output buffering och stäng av felvisning för att säkerställa ren bildoutput
ob_start();
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);

require_once __DIR__ . '/config.php';
require_once __DIR__ . '/security.php';
require_once __DIR__ . '/lib/participant_helper.php';
require_once __DIR__ . '/logger.php';

// Rensa output buffer (tar bort eventuella varningar)
ob_clean();

// Säkerhetsheaders (men inte Content-Security-Policy som kan blockera bildladdning)
header('X-Content-Type-Options: nosniff');
header('X-Frame-Options: DENY');
header('X-XSS-Protection: 1; mode=block');

// Hämta parametrar
$projectId = $_GET['project_id'] ?? '';
$fileName = $_GET['file'] ?? '';

logDebug('view_file', 'Request received', [
    'project_id' => $projectId,
    'file' => $fileName,
    'get_params' => $_GET
]);

if (empty($projectId) || empty($fileName)) {
    logWarning('view_file', 'Missing parameters', [
        'project_id' => $projectId,
        'file' => $fileName
    ]);
    http_response_code(400);
    exit('Ogiltiga parametrar');
}

// Säkerställ att project_id är säkert (förhindra directory traversal)
$originalProjectId = $projectId;
$projectId = sanitizeInput($projectId, 'project_id');
if (empty($projectId) || !preg_match('/^[a-zA-Z0-9_-]+$/', $projectId)) {
    logWarning('view_file', 'Invalid project_id', [
        'original' => $originalProjectId,
        'sanitized' => $projectId
    ]);
    http_response_code(400);
    exit('Ogiltigt projekt-ID');
}

// Säkerställ att filnamnet är säkert (förhindra directory traversal)
// Använd basename för att ta bort eventuella sökvägar
$originalFileName = $fileName;
$fileName = basename($fileName);

// Validera filnamn: tillåt vanliga filnamn-tecken inklusive mellanslag och specialtecken
// Men blockera path separators (/ \), null bytes, och kontrolltecken för säkerhet
// basename() har redan tagit bort path-komponenter, så vi behöver bara blockera farliga tecken
if (empty($fileName) || preg_match('/[\x00\/\\]/', $fileName)) {
    logWarning('view_file', 'Invalid filename - contains dangerous characters', [
        'original' => $originalFileName,
        'basename' => $fileName
    ]);
    http_response_code(400);
    exit('Ogiltigt filnamn');
}

// Ytterligare säkerhetskontroll: blockera filnamn som börjar med punkt (dolda filer) eller innehåller ".."
// (även om basename() borde ha tagit bort dessa, är det extra säkerhet)
if ($fileName === '.' || $fileName === '..' || strpos($fileName, '..') !== false) {
    logWarning('view_file', 'Invalid filename - directory traversal attempt', [
        'original' => $originalFileName,
        'basename' => $fileName
    ]);
    http_response_code(400);
    exit('Ogiltigt filnamn');
}

// Konstruera filsökväg
$filePath = __DIR__ . '/' . rtrim(UPLOAD_ROOT, '/\\') . '/' . $projectId . '/' . $fileName;

logDebug('view_file', 'Checking file', [
    'file_path' => $filePath,
    'project_id' => $projectId,
    'file_name' => $fileName,
    'upload_root' => UPLOAD_ROOT,
    'file_exists' => file_exists($filePath),
    'is_file' => is_file($filePath)
]);

// Kontrollera att filen finns
if (!file_exists($filePath) || !is_file($filePath)) {
    logError('view_file', 'File not found', [
        'file_path' => $filePath,
        'project_id' => $projectId,
        'file_name' => $fileName,
        'upload_root' => UPLOAD_ROOT,
        'directory_exists' => is_dir(dirname($filePath)),
        'directory_contents' => is_dir(dirname($filePath)) ? scandir(dirname($filePath)) : null
    ]);
    http_response_code(404);
    exit('Filen hittades inte: ' . htmlspecialchars($fileName));
}

// Kontrollera att filen ligger inom uploads-katalogen (extra säkerhet mot directory traversal)
$realFilePath = realpath($filePath);
$realUploadsPath = realpath(__DIR__ . '/' . rtrim(UPLOAD_ROOT, '/\\'));
if (!$realFilePath || !$realUploadsPath || strpos($realFilePath, $realUploadsPath) !== 0) {
    logError('view_file', 'Security check failed - file outside uploads directory', [
        'real_file_path' => $realFilePath,
        'real_uploads_path' => $realUploadsPath,
        'file_path' => $filePath
    ]);
    http_response_code(403);
    exit('Otillåten åtkomst');
}

// Kontrollera filtyp och sätt rätt MIME-typ
$finfo = new finfo(FILEINFO_MIME_TYPE);
$mimeType = $finfo->file($filePath);

// Säkerställ att filen är en tillåten typ
$extension = strtolower(pathinfo($fileName, PATHINFO_EXTENSION));
$allowedExtensions = ['jpg', 'jpeg', 'png', 'gif', 'webp', 'heic', 'heif', 'mp4', 'webm', 'mov', 'avi', 'pdf', 'txt', 'csv'];

if (!in_array($extension, $allowedExtensions)) {
    logWarning('view_file', 'File type not allowed', [
        'extension' => $extension,
        'file_name' => $fileName,
        'allowed_extensions' => $allowedExtensions
    ]);
    http_response_code(403);
    exit('Otillåten filtyp');
}

// Sätt headers för visning/nedladdning
header('Content-Type: ' . $mimeType);
header('Content-Length: ' . filesize($filePath));
header('Cache-Control: public, max-age=3600'); // Cache i 1 timme för bilder
header('X-Content-Type-Options: nosniff');

// För PDF och vissa andra filtyper, tvinga nedladdning
if (in_array($extension, ['pdf', 'txt', 'csv'])) {
    header('Content-Disposition: attachment; filename="' . basename($fileName) . '"');
} else {
    // För bilder och videor, visa inline
    header('Content-Disposition: inline; filename="' . basename($fileName) . '"');
}

// Logga framgångsrik filservering
logInfo('view_file', 'File served successfully', [
    'file_name' => $fileName,
    'project_id' => $projectId,
    'mime_type' => $mimeType,
    'file_size' => filesize($filePath),
    'extension' => $extension
]);

// Rensa output buffer innan vi skickar bilden
ob_clean();

// Outputa filen
readfile($filePath);

// Stäng output buffering och skicka allt
ob_end_flush();
exit;
?>
