<?php
require_once 'config.php';
require_once __DIR__ . '/lib/participant_helper.php';
require_once __DIR__ . '/lib/mapping_helper.php';
require_once __DIR__ . '/logger.php';

// Hämta projekt-ID från URL
$projectId = $_GET['project_id'] ?? '';

// Hämta user_id från URL eller session (fungerar för både users och staff)
$participantId = $_GET['user_id'] ?? $_SESSION['user_id'] ?? '';

// Spara user_id i session om det finns i URL
if (!empty($_GET['user_id'])) {
    $_SESSION['user_id'] = $_GET['user_id'];
    $participantId = $_GET['user_id'];
}

$participantType = getParticipantType($participantId);

// Hämta projektdata
$projectsData = [];
$projectNames = [];
$archivedMode = isset($_GET['archived']) && $_GET['archived'] == '1';
if (file_exists('data/projects.json')) {
    $projectsData = json_decode(file_get_contents('data/projects.json'), true) ?: [];
    // Filtrera om vi är i arkiverat läge
    if ($archivedMode) {
        $projectsData = array_values(array_filter($projectsData, function($p) {
            return ($p['status'] ?? 'active') === 'archived';
        }));
    }
    // Sortera projekt alfabetiskt efter namn
    usort($projectsData, function($a, $b) {
        return strcasecmp($a['name'] ?? $a['id'] ?? '', $b['name'] ?? $b['id'] ?? '');
    });
    foreach ($projectsData as $project) {
        $projectNames[$project['id']] = $project['name'];
    }
}

// Hämta användardata
$userData = [];
if (file_exists('data/userids.json')) {
    $userContent = file_get_contents('data/userids.json');
    if ($userContent !== false) {
        $userData = json_decode($userContent, true) ?: [];
    }
}

// Funktion för att hämta användarnamn
function getUserFullName($userId) {
    global $userData;
    foreach ($userData as $user) {
        if ($user['user_id'] === $userId) {
            return $user['full_name'];
        }
    }
    return '';
}

// Formatera filstorlek som "3.4M" (en decimal)
function formatSizeM($bytes) {
    if (!is_numeric($bytes) || $bytes <= 0) {
        return '0.0M';
    }
    $mb = $bytes / (1024 * 1024);
    return number_format($mb, 1, '.', '') . 'M';
}

// Formatera filstorlek smart (KB för små filer, MB för stora)
function formatSize($bytes) {
    if (!is_numeric($bytes) || $bytes <= 0) {
        return '0B';
    }
    if ($bytes < 1024 * 1024) {
        // Mindre än 1MB, visa i KB
        $kb = $bytes / 1024;
        return number_format($kb, 0) . 'K';
    } else {
        // 1MB eller mer, visa i MB
        $mb = $bytes / (1024 * 1024);
        return number_format($mb, 1, '.', '') . 'M';
    }
}

// Hämta bilder för valt projekt
$projectImages = [];
$selectedProjectName = '';
$selectedProjectDescription = '';
// Kontrollera om användaren/staff är deltagare i projektets chatt
$isChatParticipant = false;
// Kontrollera om användaren är admin
$isAdmin = false;
$isInternalUser = false;
$userType = null;
if (!empty($participantId)) {
    $isAdmin = isParticipantAdmin($participantId);
    $participantType = getParticipantType($participantId);
    // Interna användare är 'user', externa är 'extern'
    $isInternalUser = ($participantType === 'user');
    
    // Hämta user_type direkt från userData
    foreach ($userData as $user) {
        if (($user['user_id'] ?? '') === $participantId) {
            $userType = $user['user_type'] ?? 'user';
            break;
        }
    }
}
if ($projectId && isset($projectNames[$projectId])) {
    $selectedProjectName = $projectNames[$projectId];
    
    // Hämta projektbeskrivning och kontrollera deltagarskap
    foreach ($projectsData as $project) {
        if ($project['id'] === $projectId) {
            $selectedProjectDescription = $project['description'] ?? '';
            
            // Kontrollera om användaren/staff är deltagare i projektets chatt
            // Admins och interna användare räknas också som deltagare (visas knappen)
            if ($isAdmin || $isInternalUser) {
                $isChatParticipant = true;
            } else {
                // För externa användare, kontrollera chatters-listan (inte participants)
                $chatters = readChatMapping($projectId);
                foreach ($chatters as $c) {
                    $cid = $c['participant_id'] ?? '';
                    if ($cid === $participantId) {
                        $isChatParticipant = true;
                        break;
                    }
                }
            }
            break;
        }
    }
    
    $projectDir = __DIR__ . '/' . UPLOAD_ROOT . '/' . $projectId;
    
    logDebug('view_project', 'Scanning project directory for images', [
        'project_id' => $projectId,
        'project_dir' => $projectDir,
        'directory_exists' => is_dir($projectDir)
    ]);
    
    if (is_dir($projectDir)) {
        $jsonFiles = glob($projectDir . '/*.json');
        
        logDebug('view_project', 'Found JSON files', [
            'project_id' => $projectId,
            'json_count' => count($jsonFiles),
            'json_files' => array_map('basename', $jsonFiles)
        ]);
        
        foreach ($jsonFiles as $jsonFile) {
            $metadata = json_decode(file_get_contents($jsonFile), true);
            if ($metadata) {
                $projectDir = dirname($jsonFile);
                $imageFile = null;
                
                // Först: Försök använda stored_file från JSON-metadata om det finns
                if (!empty($metadata['stored_file'])) {
                    $storedFile = $projectDir . '/' . basename($metadata['stored_file']);
                    if (file_exists($storedFile) && is_file($storedFile)) {
                        $imageFile = $storedFile;
                        logDebug('view_project', 'Using stored_file from JSON metadata', [
                            'json_file' => basename($jsonFile),
                            'stored_file' => $metadata['stored_file'],
                            'image_file' => basename($imageFile)
                        ]);
                    }
                }
                
                // Fallback: Om stored_file inte finns eller inte matchar, försök hitta baserat på JSON-filnamn
                if (!$imageFile) {
                    $baseName = str_replace('.json', '', $jsonFile);
                    
                    // Kontrollera alla stödda bildformat och videoformat
                    $supportedImageFormats = ['jpg', 'jpeg', 'png', 'webp', 'heic', 'heif'];
                    $supportedVideoFormats = ['mp4', 'webm', 'mov', 'avi'];
                    $allSupportedFormats = array_merge($supportedImageFormats, $supportedVideoFormats);
                    
                    logDebug('view_project', 'Looking for image file (fallback)', [
                        'json_file' => basename($jsonFile),
                        'base_name' => basename($baseName),
                        'formats_to_check' => $allSupportedFormats
                    ]);
                    
                    foreach ($allSupportedFormats as $format) {
                        $testFile = $baseName . '.' . $format;
                        $testFileExists = file_exists($testFile);
                        logDebug('view_project', 'Testing file format', [
                            'json_file' => basename($jsonFile),
                            'test_file' => basename($testFile),
                            'full_path' => $testFile,
                            'format' => $format,
                            'exists' => $testFileExists
                        ]);
                        if ($testFileExists) {
                            $imageFile = $testFile;
                            logDebug('view_project', 'Found matching image file', [
                                'json_file' => basename($jsonFile),
                                'image_file' => basename($imageFile),
                                'format' => $format
                            ]);
                            break;
                        }
                    }
                }
                
                if ($imageFile) {
                    $metadata['image_path'] = $imageFile;
                    $metadata['image_url'] = 'view_file.php?project_id=' . urlencode($projectId) . '&file=' . urlencode(basename($imageFile));
                    
                    // Bestäm filtyp (använd från JSON om det finns, annars gissa från filändelse)
                    if (isset($metadata['file_type'])) {
                        // Använd file_type från JSON
                    } else {
                        $fileExtension = strtolower(pathinfo($imageFile, PATHINFO_EXTENSION));
                        $supportedVideoFormats = ['mp4', 'webm', 'mov', 'avi'];
                        $isVideo = in_array($fileExtension, $supportedVideoFormats);
                        $metadata['file_type'] = $isVideo ? 'video' : 'image';
                    }
                    
                    // Lägg till thumb_url (använd thumb_file från JSON om det finns, annars gissa)
                    if (($metadata['file_type'] ?? 'image') !== 'video') {
                        if (!empty($metadata['thumb_file'])) {
                            $thumbPath = $projectDir . '/' . basename($metadata['thumb_file']);
                            if (file_exists($thumbPath)) {
                                $metadata['thumb_path'] = $thumbPath;
                                $metadata['thumb_url'] = 'view_file.php?project_id=' . urlencode($projectId) . '&file=' . urlencode(basename($thumbPath));
                            }
                        } else {
                            // Fallback: Gissa thumbnail-namn
                            $baseNoExt = preg_replace('/\.(jpg|jpeg|png|webp|heic|heif)$/i', '', $imageFile);
                            $thumbPath = $baseNoExt . '-thumb.jpg';
                            if (file_exists($thumbPath)) {
                                $metadata['thumb_path'] = $thumbPath;
                                $metadata['thumb_url'] = 'view_file.php?project_id=' . urlencode($projectId) . '&file=' . urlencode(basename($thumbPath));
                            }
                        }
                    }
                    
                    logDebug('view_project', 'Adding image to projectImages array', [
                        'image_url' => $metadata['image_url'],
                        'thumb_url' => $metadata['thumb_url'] ?? 'none',
                        'file_type' => $metadata['file_type'] ?? 'unknown',
                        'image_path' => basename($imageFile),
                        'stored_file' => $metadata['stored_file'] ?? 'not in JSON'
                    ]);
                    
                    $projectImages[] = $metadata;
                } else {
                    logWarning('view_project', 'JSON file found but no matching image file', [
                        'json_file' => basename($jsonFile),
                        'base_name' => basename($baseName),
                        'project_id' => $projectId
                    ]);
                }
            } else {
                logWarning('view_project', 'Invalid JSON metadata', [
                    'json_file' => basename($jsonFile),
                    'project_id' => $projectId
                ]);
            }
        }
        
        logInfo('view_project', 'Image scan complete', [
            'project_id' => $projectId,
            'images_found' => count($projectImages)
        ]);
        
        // Sortera efter datum (nyaste först)
        usort($projectImages, function($a, $b) {
            return strtotime($b['stored_at']) - strtotime($a['stored_at']);
        });
    }
}

// Hämta dokument för valt projekt
$projectDocuments = [];
if ($projectId && isset($projectNames[$projectId])) {
    $projectDir = __DIR__ . '/' . UPLOAD_ROOT . '/' . $projectId;
    
    if (is_dir($projectDir)) {
        $entries = scandir($projectDir) ?: [];
        
        // Bildformat och videoformat att filtrera bort från dokumentlistan
        $imageExts = ['jpg','jpeg','png','webp','heic','heif'];
        $videoExts = ['mp4','webm','mov','avi'];
        $mediaExts = array_merge($imageExts, $videoExts);
        
        foreach ($entries as $entry) {
            if ($entry === '.' || $entry === '..') continue;
            $fullPath = $projectDir . DIRECTORY_SEPARATOR . $entry;
            if (!is_file($fullPath)) continue;
            
            $ext = strtolower(pathinfo($entry, PATHINFO_EXTENSION));
            
            // Hoppa över bildfiler, videofiler, thumbnails och media-metadata
            if (in_array($ext, $mediaExts, true)) continue;
            if (str_ends_with($entry, '-thumb.jpg')) continue;
            if ($ext === 'json') continue;
            
            // Hoppa över metadatafilerna (.finfo) i huvudlistan
            if ($ext === 'finfo') continue;
            
            // För dokument (t.ex. pdf, txt)
            $metaPath = preg_replace('/\.[^.]+$/', '', $fullPath) . '.finfo';
            $meta = [];
            if (is_file($metaPath)) {
                $metaRaw = file_get_contents($metaPath);
                $meta = json_decode($metaRaw, true) ?: [];
            }
            
            $projectDocuments[] = [
                'name' => $entry,
                'path' => $fullPath,
                'url' => 'view_file.php?project_id=' . urlencode($projectId) . '&file=' . urlencode($entry),
                'bytes' => isset($meta['bytes']) ? (int)$meta['bytes'] : (@filesize($fullPath) ?: 0),
                'stored_at' => $meta['stored_at'] ?? date('c', @filemtime($fullPath) ?: time()),
                'note' => $meta['note'] ?? '',
                'mime' => $meta['mime'] ?? null,
                'original' => $meta['original'] ?? $entry,
                'meta' => $meta, // Spara hela metadata-arrayen
            ];
        }
        
        // Sortera dokument efter datum (nyaste först)
        usort($projectDocuments, function($a, $b) {
            return strtotime($b['stored_at']) - strtotime($a['stored_at']);
        });
    }
}
?>
<!DOCTYPE html>
<html lang="sv">
<head>
    <meta charset="utf-8" />
    <meta name="viewport" content="width=device-width,initial-scale=1" />
    <title>Visa projekt - ByggArkiv</title>
    <style>
        body { 
            font-family: system-ui, -apple-system, Segoe UI, Roboto, sans-serif; 
            margin: 16px; 
            background-color: #f5f5f5;
            line-height: 1.6;
        }
        .header {
            text-align: center;
            margin-bottom: 30px;
        }
        .header img {
            max-width: 100%;
            height: auto;
            max-height: 120px;
            display: block;
            margin: 0 auto;
        }
        .container {
            max-width: 1200px;
            margin: 0 auto;
        }
        .project-selector {
            background: white;
            border-radius: 12px;
            padding: 20px;
            margin-bottom: 24px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
            text-align: center;
        }
        .project-selector select {
            padding: 12px;
            border: 1px solid #ccc;
            border-radius: 8px;
            font-size: 16px;
            background: white;
            min-width: 300px;
            margin: 0 10px;
        }
        .project-selector select:focus {
            outline: none;
            border-color: #1a73e8;
            box-shadow: 0 0 0 2px rgba(26, 115, 232, 0.2);
        }
        .project-info {
            background: white;
            border-radius: 12px;
            padding: 20px;
            margin-bottom: 24px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
            text-align: center;
        }
        .project-title {
            font-size: 24px;
            font-weight: 600;
            color: #1a73e8;
            margin-bottom: 8px;
        }
        .project-count {
            color: #666;
            font-size: 16px;
        }
        .results-section {
            background: white;
            border-radius: 12px;
            box-shadow: 0 2px 8px rgba(0,0,0,0.1);
            overflow: hidden;
        }
        .results-header {
            background: #1a73e8;
            color: white;
            padding: 16px 20px;
            font-size: 18px;
            font-weight: 600;
        }
        .results-content {
            padding: 20px;
        }
        .file-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
            gap: 20px;
        }
        .file-item {
            border: 1px solid #e0e0e0;
            border-radius: 8px;
            overflow: hidden;
            background: #fafafa;
            transition: transform 0.2s, box-shadow 0.2s;
            position: relative;
        }
        .file-item:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(0,0,0,0.15);
        }
        .file-image {
            width: 100%;
            height: 200px;
            object-fit: cover;
            background: #f0f0f0;
            cursor: pointer;
            transition: opacity 0.2s;
        }
        .file-image:hover {
            opacity: 0.9;
        }
        .file-info {
            padding: 16px;
        }
        .file-meta {
            font-size: 12px;
            color: #666;
            display: flex;
            flex-direction: column;
            gap: 4px;
            margin-bottom: -4px;
        }
        .file-meta-row {
            display: flex;
            justify-content: space-between;
            align-items: center;
            gap: 8px;
        }
        .file-date {
            flex: 1;
        }
        .file-user-id {
            color: #666;
            font-weight: normal;
        }
        .file-filename {
            font-size: 12px;
            color: #1a73e8;
            text-decoration: underline;
            word-break: break-all;
        }
        .file-note {
            margin-top: 6px;
            padding: 6px 8px 2px 8px;
            background: #e3f2fd;
            border-radius: 4px;
            font-style: normal;
            font-weight: normal;
            color: #1976d2;
            margin-bottom: 2px;
            word-wrap: break-word;
        }
        
        .edit-btn {
            position: absolute;
            top: 8px;
            right: 8px;
            background: white;
            color: #ff9800;
            border: 2px solid #ff9800;
            border-radius: 50%;
            width: 30px;
            height: 30px;
            font-size: 16px;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            opacity: 0.9;
            transition: opacity 0.2s;
            z-index: 10;
            pointer-events: auto;
        }
        .edit-btn:hover {
            opacity: 1;
            background: #ff9800;
            color: white;
        }
        .no-results {
            text-align: center;
            padding: 40px;
            color: #666;
            font-style: italic;
        }
        .back-link {
            text-align: center;
            margin-top: 20px;
        }
        .back-link a {
            color: #1a73e8;
            text-decoration: none;
            font-weight: 500;
        }
        .back-link a:hover {
            text-decoration: underline;
        }
        /* Lightbox Modal Styles */
        .lightbox {
            display: none;
            position: fixed;
            z-index: 1000;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0, 0, 0, 0.9);
            backdrop-filter: blur(5px);
        }
        
        .lightbox-content {
            position: relative;
            margin: auto;
            padding: 20px;
            width: 90%;
            height: 90%;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        
        .lightbox-image {
            max-width: 100%;
            max-height: 100%;
            object-fit: contain;
            border-radius: 8px;
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.5);
        }
        
        .lightbox-close {
            position: absolute;
            top: 20px;
            right: 35px;
            color: #f1f1f1;
            font-size: 40px;
            font-weight: bold;
            cursor: pointer;
            z-index: 1001;
            background: rgba(0, 0, 0, 0.5);
            border-radius: 50%;
            width: 50px;
            height: 50px;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: background-color 0.2s;
        }
        
        .lightbox-close:hover {
            background: rgba(0, 0, 0, 0.8);
        }
        
        .lightbox-info {
            position: absolute;
            bottom: 20px;
            left: 20px;
            right: 20px;
            background: rgba(0, 0, 0, 0.8);
            color: white;
            padding: 15px;
            border-radius: 8px;
            font-size: 14px;
        }
        
        .lightbox-filename {
            font-weight: bold;
            margin-bottom: 5px;
        }
        
        .lightbox-meta {
            display: flex;
            justify-content: space-between;
            flex-wrap: wrap;
            gap: 10px;
        }
        
        @media (max-width: 768px) {
            .file-grid {
                grid-template-columns: 1fr;
            }
            .project-selector select {
                min-width: 250px;
                margin: 10px 0;
            }
            .lightbox-content {
                width: 95%;
                height: 95%;
                padding: 10px;
            }
            .lightbox-close {
                top: 10px;
                right: 20px;
                font-size: 30px;
                width: 40px;
                height: 40px;
            }
            .lightbox-info {
                bottom: 10px;
                left: 10px;
                right: 10px;
                padding: 10px;
                font-size: 12px;
            }
            .lightbox-meta {
                flex-direction: column;
                gap: 5px;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <header class="header">
            <a href="index.html">
                <img src="data/logo.png" alt="ByggArkiv Snickare Stefan" class="logo">
            </a>
        </header>

        <div class="back-link" style="text-align: center; margin-bottom: 20px;">
            <a href="index.html<?= !empty($_GET['user_id']) ? '?user_id=' . htmlspecialchars($_GET['user_id']) : '' ?>">← Tillbaka till uppladdning</a>
        </div>

        <?php if ($projectId && $selectedProjectName): ?>
            <?php 
            // Kontrollera om projektet är arkiverat
            $isArchived = false;
            foreach ($projectsData as $project) {
                if ($project['id'] === $projectId && ($project['status'] ?? 'active') === 'archived') {
                    $isArchived = true;
                    break;
                }
            }
            ?>
            <div class="results-section" <?= $isArchived ? 'style="background: #f8d7da; border-color: #dc3545;"' : '' ?>>
                <div class="results-header" <?= $isArchived ? 'style="background: #dc3545;"' : '' ?> style="display: flex; justify-content: space-between; align-items: center;">
                    <div>
                        <span id="fileCountDisplay"><?= htmlspecialchars($selectedProjectName) ?> (<?= htmlspecialchars($projectId) ?>) - <?= count($projectImages) ?> filer</span>
                        <?php if (!empty($selectedProjectDescription)): ?>
                            <div style="font-size: 12px; margin-top: 4px; opacity: 0.9;">
                                <?= htmlspecialchars($selectedProjectDescription) ?>
                            </div>
                        <?php endif; ?>
                    </div>
                    <?php if ($isChatParticipant && $projectId): ?>
                    <a href="chat/add_participant_and_redirect.php?project_id=<?= htmlspecialchars($projectId) ?><?= !empty($participantId) ? '&user_id=' . htmlspecialchars($participantId) : '' ?>" 
                       style="display: inline-flex; align-items: center; justify-content: center; width: 36px; height: 36px; background: rgba(255, 255, 255, 0.2); color: white; text-decoration: none; border-radius: 8px; transition: all 0.2s; border: 1px solid rgba(255, 255, 255, 0.3);" 
                       title="Öppna projektchatt"
                       onmouseover="this.style.background='rgba(255, 255, 255, 0.3)'; this.style.borderColor='rgba(255, 255, 255, 0.5)'" 
                       onmouseout="this.style.background='rgba(255, 255, 255, 0.2)'; this.style.borderColor='rgba(255, 255, 255, 0.3)'">
                        <svg width="20" height="20" viewBox="0 0 24 24" fill="currentColor">
                            <path d="M20 2H4c-1.1 0-2 .9-2 2v18l4-4h14c1.1 0 2-.9 2-2V4c0-1.1-.9-2-2-2z"/>
                        </svg>
                    </a>
                    <?php endif; ?>
                </div>
                <div class="results-content">
                    <?php if (empty($projectImages)): ?>
                        <div class="no-results">
                        Inga filer uppladdade än
                        </div>
                    <?php else: ?>
                        <div class="search-filter-container" style="margin-bottom: 20px;">
                            <div style="display: flex; align-items: center; gap: 10px; max-width: 400px;">
                                <input type="text" 
                                       id="imageNumberSearch" 
                                       placeholder="Sök efter bildnummer..." 
                                       style="flex: 1; padding: 10px; border: 1px solid #ccc; border-radius: 8px; font-size: 14px;">
                                <button id="clearSearchBtn" 
                                        onclick="clearImageNumberSearch()" 
                                        style="display: none; background: #dc3545; color: white; border: none; border-radius: 50%; width: 32px; height: 32px; cursor: pointer; font-size: 18px; line-height: 1; opacity: 0.8; transition: opacity 0.2s;"
                                        onmouseover="this.style.opacity='1'" 
                                        onmouseout="this.style.opacity='0.8'"
                                        title="Rensa sökning">×</button>
                            </div>
                        </div>
                        <div class="file-grid" id="fileGrid">
                            <?php foreach ($projectImages as $image): ?>
                                <?php 
                                logDebug('view_project', 'Rendering image in HTML', [
                                    'image_url' => $image['image_url'] ?? 'missing',
                                    'thumb_url' => $image['thumb_url'] ?? 'missing',
                                    'file_type' => $image['file_type'] ?? 'unknown',
                                    'image_path' => basename($image['image_path'] ?? 'unknown')
                                ]);
                                
                                $uid = $image['user_id'] ?? '';
                                $uname = getUserFullName($uid);
                                $displayUser = $uname !== '' ? htmlspecialchars($uname) : 'Ingen användare';
                                $displayDate = date('Y-m-d H:i', strtotime($image['stored_at']));
                                $filename = basename($image['image_path']);
                                $sizeDisplay = '';
                                if (!empty($image['image_path']) && file_exists($image['image_path'])) {
                                    $sizeBytes = @filesize($image['image_path']);
                                    if ($sizeBytes !== false) {
                                        $sizeDisplay = ' (' . formatSizeM($sizeBytes) . ')';
                                    }
                                }
                                $note = !empty($image['note']) ? htmlspecialchars($image['note']) : '';
                                $isVideo = ($image['file_type'] ?? 'image') === 'video';
                                $imageNumber = isset($image['image_number']) && is_numeric($image['image_number']) ? (int)$image['image_number'] : null;
                                $imageNumberText = $imageNumber !== null ? ($isVideo ? 'Video #' . $imageNumber : 'Bild #' . $imageNumber) : '';
                                ?>
                                <div class="file-item" 
                                     data-image-path="<?= htmlspecialchars($image['image_path']) ?>"
                                     data-json-path="<?= htmlspecialchars(str_replace(['.jpg', '.png', '.heic', '.heif', '.mp4', '.webm', '.mov', '.avi'], '.json', $image['image_path'])) ?>"
                                     data-image-number="<?= $imageNumber !== null ? htmlspecialchars((string)$imageNumber) : '' ?>">
                                    <?php if (empty($image['note'])): ?>
                                        <button class="edit-btn" onclick="addDescription(this)" title="Ändra beskrivning">
                                            <svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor">
                                                <path d="M3 17.25V21h3.75L17.81 9.94l-3.75-3.75L3 17.25zM20.71 7.04c.39-.39.39-1.02 0-1.41l-2.34-2.34c-.39-.39-1.02-.39-1.41 0l-1.83 1.83 3.75 3.75 1.83-1.83z"/>
                                            </svg>
                                        </button>
                                    <?php endif; ?>
                                    <?php if ($isVideo): ?>
                                        <video class="file-image" 
                                               controls 
                                               preload="metadata"
                                               onclick="openLightbox('<?= htmlspecialchars($image['image_url']) ?>', '<?= htmlspecialchars($filename) ?>', '<?= htmlspecialchars($displayDate) ?>', '<?= htmlspecialchars($displayUser) ?>', '<?= htmlspecialchars($note) ?>', '<?= htmlspecialchars($imageNumberText) ?>')">
                                            <source src="<?= htmlspecialchars($image['image_url']) ?>" type="<?= htmlspecialchars($image['mime'] ?? 'video/mp4') ?>">
                                            Din webbläsare stöder inte videouppspelning.
                                        </video>
                                    <?php else: ?>
                                        <img src="<?= htmlspecialchars($image['thumb_url'] ?? $image['image_url']) ?>" 
                                             alt="Bild" 
                                             class="file-image"
                                             loading="lazy"
                                             onerror="console.error('Image load error:', this.src); this.style.display='none'"
                                             onload="console.log('Image loaded successfully:', this.src)"
                                             onclick="openLightbox('<?= htmlspecialchars($image['image_url']) ?>', '<?= htmlspecialchars($filename) ?>', '<?= htmlspecialchars($displayDate) ?>', '<?= htmlspecialchars($displayUser) ?>', '<?= htmlspecialchars($note) ?>', '<?= htmlspecialchars($imageNumberText) ?>')">
                                    <?php endif; ?>
                                    <div class="file-info">
                                        <?php if ($imageNumberText): ?>
                                            <div class="file-image-number" style="font-weight: bold; color: #1976d2; margin-bottom: 4px;">
                                                <?= htmlspecialchars($imageNumberText) ?>
                                            </div>
                                        <?php endif; ?>
                                        <div class="file-meta">
                                            <div class="file-meta-row">
                                                <span class="file-date"><?= $displayDate ?><?= $sizeDisplay ?></span>
                                                <span class="file-user-id">
                                                    <?= $displayUser ?>
                                                </span>
                                            </div>
                                        </div>
                                        <a href="<?= htmlspecialchars($image['image_url']) ?>" class="file-filename" download>
                                            <?= $filename ?>
                                        </a>
                                        <?php if (!empty($image['note'])): ?>
                                            <div class="file-note">
                                                <?= $note ?>
                                            </div>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
            
            <!-- Dokumentsektion -->
            <?php if (!empty($projectDocuments)): ?>
                <div id="documents-section" class="results-section" style="margin-top: 24px;" <?= $isArchived ? 'style="margin-top: 24px; background: #f8d7da; border-color: #dc3545;"' : '' ?>>
                    <div class="results-header" <?= $isArchived ? 'style="background: #dc3545;"' : '' ?>>
                        <span>Dokument (<?= count($projectDocuments) ?>)</span>
                    </div>
                    <div class="results-content">
                        <div class="file-grid">
                            <?php foreach ($projectDocuments as $doc): ?>
                                <?php 
                                $displayDate = date('Y-m-d H:i', strtotime($doc['stored_at']));
                                $sizeDisplay = '';
                                if ($doc['bytes'] > 0) {
                                    $sizeDisplay = ' (' . formatSize($doc['bytes']) . ')';
                                }
                                $note = !empty($doc['note']) ? htmlspecialchars($doc['note']) : '';
                                $fileIcon = $doc['mime'] === 'application/pdf' ? '📄' : '📝';
                                $originalName = !empty($doc['original']) ? htmlspecialchars($doc['original']) : htmlspecialchars($doc['name']);
                                ?>
                                <div class="file-item">
                                    <div class="file-info">
                                        <!-- Rad 1: Originalfilnamn som klickbar länk -->
                                        <a href="<?= htmlspecialchars($doc['url']) ?>" class="file-filename" target="_blank" rel="noopener" style="text-decoration: underline; word-break: break-all; font-size: 16px; font-weight: 500;">
                                            <?= $originalName ?>
                                        </a>
                                        
                                        <!-- Rad 2: Datum till vänster, storlek till höger -->
                                        <div class="file-meta">
                                            <div class="file-meta-row">
                                                <span class="file-date"><?= $displayDate ?></span>
                                                <span style="color: #666;"><?= $sizeDisplay ?></span>
                                            </div>
                                        </div>
                                        
                                        <!-- Rad 3: Kommentar -->
                                        <?php if (!empty($doc['note'])): ?>
                                            <div class="file-note">
                                                <?= $note ?>
                                            </div>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        </div>
                    </div>
                </div>
            <?php endif; ?>
        <?php else: ?>
            <div class="results-section">
                <div class="results-header">
                    <span>Välj ett projekt för att visa bilder</span>
                </div>
                <div class="results-content">
                    <div class="no-results">
                        Använd dropdown-menyn ovan för att välja ett projekt.
                    </div>
                </div>
            </div>
        <?php endif; ?>

        </div>
    </div>

    <!-- Lightbox Modal -->
    <div id="lightbox" class="lightbox">
        <div class="lightbox-content">
            <span class="lightbox-close" onclick="closeLightbox()">&times;</span>
            <img id="lightbox-image" class="lightbox-image" src="" alt="" style="display: none;">
            <video id="lightbox-video" class="lightbox-image" controls style="display: none;">
                <source src="" type="video/mp4">
                Din webbläsare stöder inte videouppspelning.
            </video>
            <div class="lightbox-info">
                <div id="lightbox-image-number" class="lightbox-filename" style="font-weight: bold; color: #fff; margin-bottom: 5px;"></div>
                <div id="lightbox-filename" class="lightbox-filename"></div>
                <div class="lightbox-meta">
                    <span id="lightbox-date"></span>
                    <span id="lightbox-user"></span>
                </div>
                <div id="lightbox-note" style="margin-top: 8px; font-style: italic;"></div>
            </div>
        </div>
    </div>

    <script>
        // Lightbox functions
        function openLightbox(imageSrc, filename, date, user, note, imageNumber) {
            const lightbox = document.getElementById('lightbox');
            const lightboxImage = document.getElementById('lightbox-image');
            const lightboxVideo = document.getElementById('lightbox-video');
            const lightboxImageNumber = document.getElementById('lightbox-image-number');
            const lightboxFilename = document.getElementById('lightbox-filename');
            const lightboxDate = document.getElementById('lightbox-date');
            const lightboxUser = document.getElementById('lightbox-user');
            const lightboxNote = document.getElementById('lightbox-note');
            
            // Bestäm om det är en video baserat på filändelse
            const isVideo = /\.(mp4|webm|mov|avi)$/i.test(filename);
            
            if (isVideo) {
                // Visa video, dölj bild
                lightboxImage.style.display = 'none';
                lightboxVideo.style.display = 'block';
                lightboxVideo.src = imageSrc;
                lightboxVideo.load(); // Ladda om videon
            } else {
                // Visa bild, dölj video
                lightboxVideo.style.display = 'none';
                lightboxImage.style.display = 'block';
                lightboxImage.src = imageSrc;
            }
            
            // Visa bildnummer om det finns
            if (imageNumber && imageNumber.trim() !== '') {
                lightboxImageNumber.textContent = imageNumber;
                lightboxImageNumber.style.display = 'block';
            } else {
                lightboxImageNumber.style.display = 'none';
            }
            
            lightboxFilename.textContent = filename;
            lightboxDate.textContent = date;
            lightboxUser.textContent = user;
            lightboxNote.textContent = note || '';
            
            lightbox.style.display = 'block';
            document.body.style.overflow = 'hidden'; // Prevent background scrolling
        }
        
        function closeLightbox() {
            const lightbox = document.getElementById('lightbox');
            const lightboxVideo = document.getElementById('lightbox-video');
            
            // Pausa video när lightbox stängs
            if (lightboxVideo) {
                lightboxVideo.pause();
            }
            
            lightbox.style.display = 'none';
            document.body.style.overflow = 'auto'; // Restore scrolling
        }
        
        // Close lightbox when clicking outside the image
        document.addEventListener('click', function(event) {
            const lightbox = document.getElementById('lightbox');
            if (event.target === lightbox) {
                closeLightbox();
            }
        });
        
        // Close lightbox with Escape key
        document.addEventListener('keydown', function(event) {
            if (event.key === 'Escape') {
                closeLightbox();
            }
        });
        

        // Hämta user_id från cookie om det inte finns i URL
        function getUserIdFromCookie() {
            const cookies = document.cookie.split(';');
            for (let cookie of cookies) {
                const [name, value] = cookie.trim().split('=');
                if (name === 'user_id') {
                    return decodeURIComponent(value);
                }
            }
            return '';
        }

        // Uppdatera länkar med user_id från cookie om det saknas
        function updateLinksWithUserId() {
            const urlParams = new URLSearchParams(window.location.search);
            const urlUserId = urlParams.get('user_id');
            
            if (!urlUserId) {
                const cookieUserId = getUserIdFromCookie();
                if (cookieUserId) {
                    // Uppdatera "Tillbaka till uppladdning"-länken
                    const backLink = document.querySelector('.back-link a');
                    if (backLink) {
                        backLink.href = `index.html?user_id=${encodeURIComponent(cookieUserId)}`;
                    }
                }
            }
        }


        // Kör när sidan laddas
        document.addEventListener('DOMContentLoaded', updateLinksWithUserId);
        
        // Scrolla till dokumentsektionen
        function scrollToDocuments() {
            const documentsSection = document.getElementById('documents-section');
            if (documentsSection) {
                documentsSection.scrollIntoView({ behavior: 'smooth', block: 'start' });
            }
        }
        
        // Scroll to top funktionalitet
        function scrollToTop() {
            window.scrollTo({
                top: 0,
                behavior: 'smooth'
            });
        }
        
        // Visa/dölj scroll-to-top knapp baserat på scroll-position
        window.addEventListener('scroll', function() {
            const scrollBtn = document.getElementById('scrollToTopBtn');
            if (window.pageYOffset > 300) {
                scrollBtn.style.display = 'block';
            } else {
                scrollBtn.style.display = 'none';
            }
        });
        
        function addDescription(button) {
            const fileItem = button.closest('.file-item');
            if (!fileItem) {
                console.error('File item not found');
                return;
            }
            
            const imagePath = fileItem.dataset.imagePath;
            const jsonPath = fileItem.dataset.jsonPath;
            
            if (!imagePath || !jsonPath) {
                console.error('Missing file paths:', { imagePath, jsonPath });
                alert('Fel: Saknade filvägar');
                return;
            }
            
            const currentDescription = prompt('Ändra beskrivning för denna bild:');
            if (currentDescription === null) {
                return; // User cancelled
            }
            
            const formData = new FormData();
            formData.append('imagePath', imagePath);
            formData.append('jsonPath', jsonPath);
            formData.append('description', currentDescription);
            
            // Hämta user_id från cookie eller URL
            const userId = getUserIdFromCookie() || new URLSearchParams(window.location.search).get('user_id') || '';
            if (userId) {
                formData.append('user_id', userId);
            }
            
            fetch('api/files/add_description.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(result => {
                if (result.ok) {
                    // Hide the edit button and add the description
                    button.style.display = 'none';
                    const noteDiv = document.createElement('div');
                    noteDiv.className = 'file-note';
                    noteDiv.style.cssText = 'margin-top: 6px; padding: 6px 8px 2px 8px; background: #e3f2fd; border-radius: 4px; font-style: normal; font-weight: normal; color: #1976d2; margin-bottom: 2px; word-wrap: break-word;';
                    noteDiv.textContent = currentDescription;
                    const fileInfo = button.closest('.file-item')?.querySelector('.file-info');
                    if (fileInfo) {
                        const filenameLink = fileInfo.querySelector('.file-filename');
                        if (filenameLink && filenameLink.parentNode === fileInfo) {
                            fileInfo.insertBefore(noteDiv, filenameLink.nextSibling);
                        } else {
                            fileInfo.appendChild(noteDiv);
                        }
                    } else {
                        button.parentNode.appendChild(noteDiv);
                    }
                } else {
                    alert('Fel: ' + result.error);
                }
            })
            .catch(error => {
                console.error('Add description error:', error);
                alert('Ett fel uppstod: ' + error.message);
            });
        }
        
        // Image number search filter functionality
        function initImageNumberSearch() {
            const imageNumberSearchInput = document.getElementById('imageNumberSearch');
            const clearSearchBtn = document.getElementById('clearSearchBtn');
            const fileCountDisplay = document.getElementById('fileCountDisplay');
            const fileGrid = document.getElementById('fileGrid');
            
            // Only initialize if search input exists (i.e., there are images)
            if (!imageNumberSearchInput || !fileGrid) {
                return;
            }
            
            // Store original count and text for display updates
            let originalCount = 0;
            let originalCountText = '';
            if (fileCountDisplay) {
                originalCountText = fileCountDisplay.textContent;
                const match = originalCountText.match(/(\d+)\s+filer/);
                if (match) {
                    originalCount = parseInt(match[1], 10);
                }
            }
            
            function filterByImageNumber() {
                const searchValue = imageNumberSearchInput.value.trim();
                const fileItems = fileGrid.querySelectorAll('.file-item');
                let visibleCount = 0;
                
                fileItems.forEach(item => {
                    const imageNumber = item.dataset.imageNumber || '';
                    
                    // If search is empty, show all items
                    if (searchValue === '') {
                        item.style.display = '';
                        visibleCount++;
                    } else {
                        // Partial matching: check if image number contains the search term
                        if (imageNumber.includes(searchValue)) {
                            item.style.display = '';
                            visibleCount++;
                        } else {
                            item.style.display = 'none';
                        }
                    }
                });
                
                // Update clear button visibility
                if (clearSearchBtn) {
                    clearSearchBtn.style.display = searchValue !== '' ? 'block' : 'none';
                }
                
                // Update count display
                if (fileCountDisplay && originalCountText) {
                    if (searchValue === '') {
                        fileCountDisplay.textContent = originalCountText;
                    } else {
                        // Update with filtered count
                        const newText = originalCountText.replace(/(\d+)\s+filer/, `${visibleCount} filer (${originalCount} totalt)`);
                        fileCountDisplay.textContent = newText;
                    }
                }
            }
            
            function clearImageNumberSearch() {
                imageNumberSearchInput.value = '';
                filterByImageNumber();
                imageNumberSearchInput.focus();
            }
            
            // Make clearImageNumberSearch available globally
            window.clearImageNumberSearch = clearImageNumberSearch;
            
            // Attach event listener for real-time filtering
            imageNumberSearchInput.addEventListener('input', filterByImageNumber);
            // Also handle Enter key to prevent form submission if in a form
            imageNumberSearchInput.addEventListener('keydown', function(e) {
                if (e.key === 'Enter') {
                    e.preventDefault();
                }
            });
            
            // Debug: Log all file items when page loads
            document.addEventListener('DOMContentLoaded', function() {
                const fileItems = fileGrid.querySelectorAll('.file-item');
                console.log('Total file items found:', fileItems.length);
                fileItems.forEach((item, index) => {
                    const img = item.querySelector('.file-image');
                    const imagePath = item.dataset.imagePath;
                    const imageNumber = item.dataset.imageNumber || 'none';
                    console.log(`File item ${index + 1}:`, {
                        imagePath: imagePath,
                        imageNumber: imageNumber,
                        hasImage: !!img,
                        imageSrc: img ? img.src : 'no image element',
                        displayStyle: window.getComputedStyle(item).display,
                        visibility: window.getComputedStyle(item).visibility
                    });
                });
            });
        }
        
        // Initialize when DOM is ready
        if (document.readyState === 'loading') {
            document.addEventListener('DOMContentLoaded', initImageNumberSearch);
        } else {
            initImageNumberSearch();
        }
    </script>
    
    <!-- Scroll to top knapp -->
    <button id="scrollToTopBtn" onclick="scrollToTop()" style="
        position: fixed;
        bottom: 30px;
        right: 30px;
        background: #1a73e8;
        color: white;
        border: none;
        border-radius: 50%;
        width: 50px;
        height: 50px;
        font-size: 20px;
        cursor: pointer;
        box-shadow: 0 4px 12px rgba(0,0,0,0.3);
        transition: all 0.3s ease;
        display: none;
        z-index: 1000;
    " onmouseover="this.style.background='#1557b0'; this.style.transform='scale(1.1)'" onmouseout="this.style.background='#1a73e8'; this.style.transform='scale(1)'" title="Scrolla till toppen">
        ↑
    </button>
</body>
</html>
